------------------------------------------------------------------------------
--                                                                          --
--                	        G N A T . RAVENSCAR                         --
--                                                                          --
--                      Copyright (C) 2008-2015, AdaCore                    --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

with System;
with Ada.Real_Time;
with GNAT.Ravenscar.Simple_Cyclic_Task;
with GNAT.Ravenscar.Simple_Sporadic_Task;
with GNAT.Ravenscar.Sporadic_Server;
with GNAT.Ravenscar.Sporadic_Server_With_Callback;
with GNAT.Ravenscar.Multiple_Queue_Sporadic_Server;
with GNAT.Ravenscar.Timed_Out_Sporadic_Server;
with GNAT.Ravenscar.Multiple_Queue_Cyclic_Server;

package Examples is

   --  a simple cyclic task
   procedure Cyclic_OP1;
   package Simple_Cyclic_Task_1 is new GNAT.Ravenscar.Simple_Cyclic_Task
     (10, 1000, 1000, Ada.Real_Time.Clock, Cyclic_OP1);

   --  a simple sporadic task
   procedure Sporadic_OP1;
   package Simple_Sporadic_Task_1 is new GNAT.Ravenscar.Simple_Sporadic_Task
    (10, 1000, Ada.Real_Time.Clock, 15, Sporadic_OP1);

   --  a sporadic server taking Param_T as input paramter to
   --  execute Sporadic_OP2
   type Param_T is
      record
         I : Integer;
         B : Boolean;
      end record;
   procedure Sporadic_OP2 (Par : Param_T);
   package Sporadic_Server_1 is new GNAT.Ravenscar.Sporadic_Server
     (10, 1000, Ada.Real_Time.Clock, 15, 4, Param_T, Sporadic_OP2);

   --  This is how to handle asynchronous invocations with OUT parameters in
   --  Ravenscar: the sporadic server invokes a procedure with both IN and OUT
   --  parameter (Sporadic_OP3); the returned value of the OUT parameter is
   --  passed to a callback operation (Sporadic_OP3_Callback)

   type Out_Param_T is
      record
         I : Integer;
      end record;
   procedure Sporadic_OP3 (In_Param : Param_T; Out_Param : out Out_Param_T);
   procedure Sporadic_OP3_Callback (Par : Out_Param_T);
   package Sporadic_Server_With_CallBack_1 is new
     GNAT.Ravenscar.Sporadic_Server_With_Callback
       (10, 1000, Ada.Real_Time.Clock, 15, 4, Param_T,
        Out_Param_T, Sporadic_OP3);

   --  A sporadic server able to fulfill multiple requests
   type Request_Kind_T is (REQ1, REQ2, REQ3);
   type Complex_Request_T (R : Request_Kind_T := Request_Kind_T'First) is
      record
         Request_Kind : Request_Kind_T := R;
         case R is
            when REQ1 =>
               null;
            when REQ2 =>
               REQ2_I : Integer;
            when REQ3 =>
               REQ3_I : Integer;
               REQ3_B : Boolean;
         end case;
      end record;

   procedure Req1;
   procedure Req2 (I : Integer);
   procedure Req3 (I : Integer; B : Boolean);
   procedure Dispatch (Req : Complex_Request_T);
   package Multiple_Queue_Sporadic_Server is new
     GNAT.Ravenscar.Multiple_Queue_Sporadic_Server
       (10, 1000, Ada.Real_Time.Clock, 15, 4,
        Request_Kind_T, Complex_Request_T, Dispatch);

   --  A timed out sporadic server: if it is not released within a given amount
   --  of time, it is automatically released by a timing event.
   procedure Timed_Out_Sporadic_OP (Par : Param_T);
   procedure Time_Out_Action;
   package Timed_Out_Sporadic_Server is new
     GNAT.Ravenscar.Timed_Out_Sporadic_Server
       (10, 1000, 2000, Ada.Real_Time.Clock, System.Any_Priority'Last,
        4, Param_T, Timed_Out_Sporadic_OP, Time_Out_Action);

   procedure Cyclic_Dispatch (Req : Complex_Request_T);
   procedure Cyclic_OP2;
   package Multiple_Queue_Cyclic_Server is new
     GNAT.Ravenscar.Multiple_Queue_Cyclic_Server
       (10, 1000, Ada.Real_Time.Clock, 200, Cyclic_OP2, 15,
        4, Request_Kind_T, Complex_Request_T, Cyclic_Dispatch);

end Examples;
