------------------------------------------------------------------------------
--                                  G P S                                   --
--                                                                          --
--                     Copyright (C) 2008-2015, AdaCore                     --
--                                                                          --
-- This library is free software;  you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software  Foundation;  either version 3,  or (at your  option) any later --
-- version. This library is distributed in the hope that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE.                            --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
------------------------------------------------------------------------------

with GNATCOLL.Templates;  use GNATCOLL.Templates;
with Ada.Text_IO;         use Ada.Text_IO;
with Asserts;             use Asserts;

procedure Templates is
   Delimiter : constant Character := '%';

   function Callback (Name : String; Quoted : Boolean) return String;
   function Callback (Name : String; Quoted : Boolean) return String is
   begin
      if Name = "version" then
         return "6.0";
      elsif Name = "^" then
         return "previous";
      elsif Name = "prev" then
         return Delimiter & "^";
      elsif Name = "1" then
         return "first";
      elsif Name = "1-" then
         return "allargs";
      elsif Name = "*" then
         return "all";
      else
         raise Invalid_Substitution;
      end if;
   end Callback;

   Substrings : Substitution_Array :=
     ((new String'("tool"), new String'("GNAT")),
      (new String'("previous"), new String'(Delimiter & "prev")));

   procedure Test
     (Str1, Str2 : String; Msg : String; Recurse : Boolean := False;
      Errors : Error_Handling := Keep_As_Is);

   procedure Test
     (Str1, Str2 : String; Msg : String; Recurse : Boolean := False;
      Errors : Error_Handling := Keep_As_Is)
   is
      Str3 : constant String := Substitute
        (Str1, Substrings, Callback'Unrestricted_Access, Delimiter, Recurse,
         Errors);
   begin
      Assert (Str2, Str3, Msg);
   end Test;

begin
   Test ("abcd", "abcd", "no substitution");
   Test ("abc%", "abc%", "delimiter is last");
   Test ("abc%%", "abc%", "doubling delimiter");
   Test ("%%abc", "%abc", "substitute at beginning of string");
   Test ("%%%%abc", "%%abc", "substitute multiple times");

   Test ("a%ret",  "a%ret",    "no substitution");
   Test ("a%previ", "a%previ", "no substitution although start matches");

   Test ("v=%version", "v=6.0", "replace version");
   Test ("v=%{version}", "v=6.0", "replace version with curly braces");
   Test ("v=%{version}a", "v=6.0a", "replace version followed by char");

   Test ("t=%tool", "t=GNAT", "replace through array");
   Test ("t=%tool%version", "t=GNAT6.0", "replace multiple times");

   Test ("t=%prev", "t=%^", "non recursive replace");
   Test ("t=%prev", "t=previous", "recursive replace", True);
   Test ("t=%previous", "t=%prev", "non recursive replace and array");
   Test ("t=%previous", "t=previous", "recursive replace and array", True);

   Test ("t=%*ab", "t=allab", "All args");
   Test ("t=%1ab", "t=firstab", "First arg");
   Test ("t=%1-ab", "t=allargsab", "All args after first");

   Test ("t=%{var:-default}", "t=default",  "Default value");

   Test ("t=%{var:=default}", "t=%{var:=default}",  "Default value",
           Errors => Keep_As_Is);
   Test ("t=%{var:=default}", "t=",  "Default value",
           Errors => Replace_With_Empty);

   Free (Substrings);
end Templates;
