/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include <tools/diagnose_ex.h>
#include <tools/stream.hxx>
#include <sal/log.hxx>
#include <vcl/vectorgraphicdata.hxx>
#include <comphelper/processfactory.hxx>
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#include <com/sun/star/graphic/PdfTools.hpp>
#include <com/sun/star/graphic/SvgTools.hpp>
#include <com/sun/star/graphic/EmfTools.hpp>
#include <com/sun/star/graphic/Primitive2DTools.hpp>
#include <com/sun/star/rendering/XIntegerReadOnlyBitmap.hpp>
#include <com/sun/star/util/XAccounting.hpp>
#include <basegfx/matrix/b2dhommatrixtools.hxx>
#include <vcl/canvastools.hxx>
#include <comphelper/seqstream.hxx>
#include <comphelper/sequence.hxx>
#include <comphelper/propertysequence.hxx>
#include <vcl/svapp.hxx>
#include <vcl/outdev.hxx>
#include <vcl/wmfexternal.hxx>
#include <vcl/pdfread.hxx>

using namespace ::com::sun::star;

BitmapEx convertPrimitive2DSequenceToBitmapEx(
    const std::deque< css::uno::Reference< css::graphic::XPrimitive2D > >& rSequence,
    const basegfx::B2DRange& rTargetRange,
    const sal_uInt32 nMaximumQuadraticPixels)
{
    BitmapEx aRetval;

    if(!rSequence.empty())
    {
        // create replacement graphic from maSequence
        // create XPrimitive2DRenderer
        try
        {
            uno::Reference< uno::XComponentContext > xContext(::comphelper::getProcessComponentContext());
            const uno::Reference< graphic::XPrimitive2DRenderer > xPrimitive2DRenderer = graphic::Primitive2DTools::create(xContext);

            uno::Sequence< beans::PropertyValue > aViewParameters;
            geometry::RealRectangle2D aRealRect;

            aRealRect.X1 = rTargetRange.getMinX();
            aRealRect.Y1 = rTargetRange.getMinY();
            aRealRect.X2 = rTargetRange.getMaxX();
            aRealRect.Y2 = rTargetRange.getMaxY();

            // get system DPI
            const Size aDPI(Application::GetDefaultDevice()->LogicToPixel(Size(1, 1), MapMode(MapUnit::MapInch)));

            const uno::Reference< rendering::XBitmap > xBitmap(
                xPrimitive2DRenderer->rasterize(
                    comphelper::containerToSequence(rSequence),
                    aViewParameters,
                    aDPI.getWidth(),
                    aDPI.getHeight(),
                    aRealRect,
                    nMaximumQuadraticPixels));

            if(xBitmap.is())
            {
                const uno::Reference< rendering::XIntegerReadOnlyBitmap> xIntBmp(xBitmap, uno::UNO_QUERY_THROW);
                aRetval = vcl::unotools::bitmapExFromXBitmap(xIntBmp);
            }
        }
        catch (const uno::Exception&)
        {
            TOOLS_WARN_EXCEPTION("vcl", "Got no graphic::XPrimitive2DRenderer!");
        }
        catch (const std::exception& e)
        {
            SAL_WARN("vcl", "Got no graphic::XPrimitive2DRenderer! : " << e.what());
        }
    }

    return aRetval;
}

static size_t estimateSize(
    std::deque<uno::Reference<graphic::XPrimitive2D>> const& rSequence)
{
    size_t nRet(0);
    for (auto& it : rSequence)
    {
        uno::Reference<util::XAccounting> const xAcc(it, uno::UNO_QUERY);
        assert(xAcc.is()); // we expect only BasePrimitive2D from SVG parser
        nRet += xAcc->estimateUsage();
    }
    return nRet;
}

bool VectorGraphicData::operator==(const VectorGraphicData& rCandidate) const
{
    if (getVectorGraphicDataType() == rCandidate.getVectorGraphicDataType())
    {
        if (getVectorGraphicDataArrayLength() == rCandidate.getVectorGraphicDataArrayLength())
        {
            if (0 == memcmp(
                getVectorGraphicDataArray().getConstArray(),
                rCandidate.getVectorGraphicDataArray().getConstArray(),
                getVectorGraphicDataArrayLength()))
            {
                return true;
            }
        }
    }

    return false;
}

void VectorGraphicData::setWmfExternalHeader(const WmfExternal& aExtHeader)
{
    if (!mpExternalHeader)
    {
        mpExternalHeader.reset( new WmfExternal );
    }

    *mpExternalHeader = aExtHeader;
}

void VectorGraphicData::ensurePdfReplacement()
{
    assert(getVectorGraphicDataType() == VectorGraphicDataType::Pdf);

    if (!maReplacement.IsEmpty())
        return; // nothing to do

    // use PDFium directly
    std::vector<Bitmap> aBitmaps;
    sal_Int32 nUsePageIndex = 0;
    if (mnPageIndex >= 0)
        nUsePageIndex = mnPageIndex;
    vcl::RenderPDFBitmaps(maVectorGraphicDataArray.getConstArray(),
                          maVectorGraphicDataArray.getLength(), aBitmaps, nUsePageIndex, 1,
                          &maSizeHint);
    if (!aBitmaps.empty())
        maReplacement = aBitmaps[0];
}

void VectorGraphicData::ensureReplacement()
{
    if (!maReplacement.IsEmpty())
        return; // nothing to do

    // shortcut for PDF - PDFium can generate the replacement bitmap for us
    // directly
    if (getVectorGraphicDataType() == VectorGraphicDataType::Pdf)
    {
        ensurePdfReplacement();
        return;
    }

    ensureSequenceAndRange();

    if (!maSequence.empty())
    {
        maReplacement = convertPrimitive2DSequenceToBitmapEx(maSequence, getRange());
    }
}

void VectorGraphicData::ensureSequenceAndRange()
{
    if (!mbSequenceCreated && maVectorGraphicDataArray.hasElements())
    {
        // import SVG to maSequence, also set maRange
        maRange.reset();

        // create Vector Graphic Data interpreter
        uno::Reference<uno::XComponentContext> xContext(::comphelper::getProcessComponentContext());

        switch (getVectorGraphicDataType())
        {
            case VectorGraphicDataType::Svg:
            {
                const uno::Reference< graphic::XSvgParser > xSvgParser = graphic::SvgTools::create(xContext);
                const uno::Reference< io::XInputStream > myInputStream(new comphelper::SequenceInputStream(maVectorGraphicDataArray));

                if (myInputStream.is())
                    maSequence = comphelper::sequenceToContainer<std::deque<css::uno::Reference< css::graphic::XPrimitive2D >>>(xSvgParser->getDecomposition(myInputStream, maPath));

                break;
            }
            case VectorGraphicDataType::Emf:
            case VectorGraphicDataType::Wmf:
            {
                const uno::Reference< graphic::XEmfParser > xEmfParser = graphic::EmfTools::create(xContext);
                const uno::Reference< io::XInputStream > myInputStream(new comphelper::SequenceInputStream(maVectorGraphicDataArray));
                uno::Sequence< ::beans::PropertyValue > aSequence;

                if (mpExternalHeader)
                {
                    aSequence = mpExternalHeader->getSequence();
                }

                if (myInputStream.is())
                {
                    // Pass the size hint of the graphic to the EMF parser.
                    geometry::RealPoint2D aSizeHint;
                    aSizeHint.X = maSizeHint.getX();
                    aSizeHint.Y = maSizeHint.getY();
                    xEmfParser->setSizeHint(aSizeHint);

                    maSequence = comphelper::sequenceToContainer<std::deque<css::uno::Reference< css::graphic::XPrimitive2D >>>(xEmfParser->getDecomposition(myInputStream, maPath, aSequence));
                }

                break;
            }
            case VectorGraphicDataType::Pdf:
            {
                const uno::Reference<graphic::XPdfDecomposer> xPdfDecomposer = graphic::PdfTools::create(xContext);
                uno::Sequence<beans::PropertyValue> aDecompositionParameters = comphelper::InitPropertySequence({
                    {"PageIndex", uno::makeAny<sal_Int32>(mnPageIndex)},
                });
                auto xPrimitive2D = xPdfDecomposer->getDecomposition(maVectorGraphicDataArray, aDecompositionParameters);
                maSequence = comphelper::sequenceToContainer<std::deque<uno::Reference<graphic::XPrimitive2D>>>(xPrimitive2D);

                break;
            }
        }

        if(!maSequence.empty())
        {
            const sal_Int32 nCount(maSequence.size());
            geometry::RealRectangle2D aRealRect;
            uno::Sequence< beans::PropertyValue > aViewParameters;

            for(sal_Int32 a(0); a < nCount; a++)
            {
                // get reference
                const css::uno::Reference< css::graphic::XPrimitive2D > xReference(maSequence[a]);

                if(xReference.is())
                {
                    aRealRect = xReference->getRange(aViewParameters);

                    maRange.expand(
                        basegfx::B2DRange(
                            aRealRect.X1,
                            aRealRect.Y1,
                            aRealRect.X2,
                            aRealRect.Y2));
                }
            }
        }
        mNestedBitmapSize = estimateSize(maSequence);
        mbSequenceCreated = true;
    }
}

auto VectorGraphicData::getSizeBytes() const -> std::pair<State, size_t>
{
    if (maSequence.empty() && maVectorGraphicDataArray.hasElements())
    {
        return std::make_pair(State::UNPARSED, maVectorGraphicDataArray.getLength());
    }
    else
    {
        return std::make_pair(State::PARSED, maVectorGraphicDataArray.getLength() + mNestedBitmapSize);
    }
}

VectorGraphicData::VectorGraphicData(
    const VectorGraphicDataArray& rVectorGraphicDataArray,
    const OUString& rPath,
    VectorGraphicDataType eVectorDataType,
    sal_Int32 nPageIndex)
:   maVectorGraphicDataArray(rVectorGraphicDataArray),
    maPath(rPath),
    mbSequenceCreated(false),
    maRange(),
    maSequence(),
    maReplacement(),
    mNestedBitmapSize(0),
    meVectorGraphicDataType(eVectorDataType),
    mnPageIndex(nPageIndex)
{
}

VectorGraphicData::VectorGraphicData(
    const OUString& rPath,
    VectorGraphicDataType eVectorDataType)
:   maVectorGraphicDataArray(),
    maPath(rPath),
    mbSequenceCreated(false),
    maRange(),
    maSequence(),
    maReplacement(),
    mNestedBitmapSize(0),
    meVectorGraphicDataType(eVectorDataType),
    mnPageIndex(-1)
{
    SvFileStream rIStm(rPath, StreamMode::STD_READ);
    if(rIStm.GetError())
        return;
    const sal_uInt32 nStmLen(rIStm.remainingSize());
    if (nStmLen)
    {
        maVectorGraphicDataArray.realloc(nStmLen);
        rIStm.ReadBytes(maVectorGraphicDataArray.begin(), nStmLen);

        if (rIStm.GetError())
        {
            maVectorGraphicDataArray = VectorGraphicDataArray();
        }
    }
}

VectorGraphicData::~VectorGraphicData()
{
}

const basegfx::B2DRange& VectorGraphicData::getRange() const
{
    const_cast< VectorGraphicData* >(this)->ensureSequenceAndRange();

    return maRange;
}

const std::deque< css::uno::Reference< css::graphic::XPrimitive2D > >& VectorGraphicData::getPrimitive2DSequence() const
{
    const_cast< VectorGraphicData* >(this)->ensureSequenceAndRange();

    return maSequence;
}

const BitmapEx& VectorGraphicData::getReplacement() const
{
    const_cast< VectorGraphicData* >(this)->ensureReplacement();

    return maReplacement;
}

BitmapChecksum VectorGraphicData::GetChecksum() const
{
    return vcl_get_checksum(0, maVectorGraphicDataArray.getConstArray(), maVectorGraphicDataArray.getLength());
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
